library(MCMCpack)
choices<-c('Rock','Scissors','Paper')


rps_game<-function(choice1,choice2){
    #function to determine winner using standard rps rules
    #choice1 and choice2 are values of R,S,P for each player respectively
    #result is win/draw/lose for player 1
    if(choice1==choice2)result<-'draw'
    else if(choice1=='Rock'&choice2=='Scissors')result<-'win'
    else if(choice1=='Paper'&choice2=='Rock')result<-'win'
    else if(choice1=='Scissors'&choice2=='Paper')result<-'win'
    else result<- 'lose'
    return(result)
}

rps_tournament<-function(n,choice_set1=choices,choice_set2=choices){
    #plays a tournament of n rounds of rps
    # choice_set1 are values of rps available to player 1
    # choice_set2 are values of rps available to player 2
    # simulated_data is container for simulations
    # cov1 and cov2 are covariates for each player, defined as random selection of rps value *not* selected
    # count is running count of simulations
    
    simulated_data<-c()
    cov1<-c()
    cov2<-c()
    
    #initialise counts at zero
    count<-matrix(0,n+1,3)
    colnames(count)<-c('win','draw','lose')
    
    #loop over games
    for(i in 1:n){
        
        #random selections from available choices
        choices1<-sample(choice_set1,1)
        choices2<-sample(choice_set2,1)
        
        #compare results
        simulated_data[i]<-rps_game(choices1,choices2)
        
        #update containers
        cov1[i]<-sample(setdiff(choices,choices1),1)
        cov2[i]<-sample(setdiff(choices,choices2),1)
        count[i+1,]<-count[i,]
        count[i+1,simulated_data[i]]<-count[i+1,simulated_data[i]]+1
    }
    
    #drop the initialisation row
    count<-count[-1,]
    
    list(data=data.frame(simulated_data=simulated_data,cov1=cov1,cov2=cov2),running_counts=count)
}

rps_tournament_changepoint<-function(n_games,choice_set1=choices,choice_set2=choices,choice_set1_change=choices,changepoint=n){
    #plays rps tournament but allows for a change in strategy of player 1
    #choice_set1 and choice_set2 are initial choices of rps for players 1 and 2
    #changepoint is number of rounds played using initial strategy
    #(currently changepoint is chosen, but could adapt code to allow this to be random)
    #choice_set1_change is set of rps available to player 1 after change at changepoint
    
    #setup containers
    simulated_data<-c()
    count<-matrix(0,n_games+1,3)
    colnames(count)<-c('win','draw','lose')
    
    #loop over matches in initial setup
    for(i in 1:changepoint){
        choices1<-sample(choice_set1,1)
        choices2<-sample(choice_set2,1)
        simulated_data[i]<-rps_game(choices1,choices2)
        count[i+1,]<-count[i,]
        count[i+1,simulated_data[i]]<-count[i+1,simulated_data[i]]+1
    }
    
    if(n_games>changepoint){
        #loop over remaining games after player 1 changes to different set of choices
        for(i in (changepoint+1):n_games){
            choices1<-sample(choice_set1_change,1)
            choices2<-sample(choice_set2,1)
            simulated_data[i]<-rps_game(choices1,choices2)
            count[i+1,]<-count[i,]
            count[i+1,simulated_data[i]]<-count[i+1,simulated_data[i]]+1
        }
    }
    #remove initialisation row
    count<-count[-1,]
    
    list(simulated_data=simulated_data,running_counts=count)
}

rps_mll<-function(x,k){
    #function to calculate (max) log lik when data are x with a change point at k
    
    #make sure x is factor so levels aren't dropped when counts are zero
    x<-factor(x,levels=c('draw','lose','win'))
    
    #calculate log lik for data up to changepoint
    x1<-x[1:min(k,length(x))]
    t1<-table(x1)
    theta1<-t1/length(x1)
    ll<-ifelse(t1[1]==0,0,t1[1]*log(theta1[1]))+ifelse(t1[2]==0,0,t1[2]*log(theta1[2]))+ifelse(t1[3]==0,0,t1[3]*log(theta1[3]))
    
    #if there are data after changepoint add log lik for such data
    if(k<length(x)){
        x2<-x[(k+1):length(x)]
        t2<-table(x2)
        theta2<-t2/length(x2)
        ll2<-ifelse(t2[1]==0,0,t2[1]*log(theta2[1]))+ifelse(t2[2]==0,0,t2[2]*log(theta2[2]))+ifelse(t2[3]==0,0,t2[3]*log(theta2[3]))
        ll<-ll+ll2
    }
    
    #return(ifelse(is.na(as.numeric(l)),-Inf,as.numeric(l)))
    return(ll)
}




rps_fit_map<-function(x,m,v,k_lim=200,plot_graph=T){
    #calculate MAP estimate of data in x for changepoint rps model
    #assumes neg bin prior on changepoint k with mean m and var v (needs m<=v) and unifrom priors for thetas
    #but note, prior on j is dnegbinom(j-1)
    
    #stop if v<m
    if(v<m)return('prior model requires v>=m')
    
    #ensure x is factor (to avoid dropped levels)
    x<-factor(x,levels<-c('draw','lose','win'))
    
    #max value of changepoint
    k_max<-length(x)
    
    #calculate size param of neg bin
    size<-m^2/(v-m)
    
    #container for loglik
    ll<-c()
    
    #calculate log lik conditioned on all possible changepoints
    for (i in 1:k_lim)ll[i]=rps_mll(x,i)
    
    #add log prior
    log_post<-ll+log(dnbinom(0:(k_lim-1),size=size,mu=m))
    
    #pick out maximum
    id<-which.max(log_post)
    
    # get corresponding estimate of theta1
    id_x<-ifelse(id>k_max,k_max,id)
    theta1<-table(x[1:id_x])/id_x
    
    #provided id is smaller than cmax, do same for theta2, otherwise return c(1/3,1/3,1/3) which is assumed prior mode
    if(id>=k_max){
        theta2<-c(1/3,1/3,1/3)
    }
    else{
        theta2<-table(x[(id+1):k_max])/(k_max-id)
    }
    
    if(plot_graph){
        #show graph of log posterior values conditioned on changepoint
        plot(log_post,pch=16)
        
        #add lines at posterior max for changepoint to plot 
        abline(v=id,col='red')
        abline(h=log_post[id],col='red')
    }
    
    return(list(map_changepoint=c(k=id,log_posterior=log_post[id]),theta1=theta1,theta2=theta2))
}



rps_fit_map_dyn<-function(x,m,v,k_lim=200){
    #fit rps_fit_map dynamically as data become available
    
    #stop if v<m
    if(v<m)return('prior model requires v>=m')
    
    #ensure x is factor (to avoid dropped levels)
    x<-factor(x,levels=c('draw','lose','win'))
    
    #max value for changepoint
    k_max<-length(x)
    
    #containers for estimates of theta based on subsets of data
    k_out<-c()
    theta1_out<-c()
    theta2_out<-c()
    theta_map<-c()
    
    #loop over all possible amounts of data
    for(i in 1:k_max){
        map<-rps_fit_map(x[1:i],m,v,k_lim=k_lim,plot=F)
        k_out<-c(k_out,map$map_changepoint["k"])
        theta1_out<-rbind(theta1_out,map$theta1)
        theta2_out<-rbind(theta2_out,map$theta2)
        if(map$map_changepoint["k"]>i)
            theta_map<-rbind(theta_map,map$theta1)
        else
            theta_map<-rbind(theta_map,map$theta2)
    }
    
    par(mfrow=c(2,2))
    plot(1:k_max,theta_map[,"win"],pch=16,xlab="index",ylab="p",main="win")
    plot(1:k_max,theta_map[,"draw"],pch=16,xlab="index",ylab="p",main="draw")
    plot(1:k_max,theta_map[,"lose"],pch=16,xlab="index",ylab="p",main="lose")
    plot(1:k_max,k_out,pch=16,xlab="index",ylab="k",main="changepoint")
    par(mfrow=c(1,1))
    
    list(theta1=theta1_out,theta2=theta2_out,k=k_out,theta_map=theta_map)
}




rps_ll<-function(k,theta1,theta2,x){
    #calculate log lik of data in x when changepoint is k and theta estimates in 2 groups are theta1 and theta2
    
    #log lik for first group
    x1<-x[1:k]
    ll<-sum(table(x1)*log(theta1))
    
    #if there is a second group, add its log lik
    if(k<length(x)){
        x2<-x[(k+1):length(x)]
        t2<-table(x2)
        ll<-ll+sum(t2*log(theta2))
    }
    
    return(ll)
}        



rps_gs<-function(x,nx=length(x),n=1000,n_burn=100,theta1_init=c(1/3,1/3,1/3),theta2_init=c(1/3,1/3,1/3),k_init=ceiling(length(x))/2,a1=c(1,1,1)*d1,a2=c(1,1,1)*d2,d1=100,d2=1,m=50,v=100,k_lim=200){
    # gibbs sampler for rps changepoint problem
    #assumes neg binom prior on changepoint (as per rps_fit)
    #now includes dirichlet priors on theta1 and theta2
    #for convenience klim specifies a maximum value for chanegepoint regardless of data
    
    #stop if v<m
    if(v<m)return('prior model requires v>=m')
    
    #ensure x is factor (to avoid dropped levels)
    x<-factor(x,levels=c('draw','lose','win'))
    
    #apply inference only to first nx observations
    x<-x[1:nx]
    
    #max value for changepoint
    k_max<-length(x)
    
    #set up containers and initialise
    theta1_out<-c()
    theta2_out<-c()
    k_out<-c()
    theta1_cur<-theta1_init
    theta2_cur<-theta2_init
    k_cur<-k_init
    a1_cur<-a1
    a2_cur<-a2
    
    # loop over udpates
    for(i in 1:(n+n_burn)){
        
        #update theta1
        x1<-x[1:k_cur]
        t1<-table(x1)
        theta1_cur<-as.vector(rdirichlet(1,a1_cur+t1))
        
        #update theta2
        #note: if changepoint beyond k_max simulate from prior
        if(k_cur<k_max){
            x2<-x[k_cur:k_max]
            t2<-table(x2)
            theta2_cur<-as.vector(rdirichlet(1,a2_cur+t2))
        }
        else{
            theta2_cur<-as.vector(rdirichlet(1,a2_cur))
        }
        
        #update changepoint
        k_prob<-k_prior(1:k_lim,m,v)
        
        #calculate conditional post on changepoint (up to proportionality)
        k_post<-exp(rps_ll_vec(1:k_lim,theta1_cur,theta2_cur,x))*k_prob
        
        #sample changepoint
        k_cur<-sample(1:k_lim,1,prob=k_post)
        
        #update containers
        theta1_out<-rbind(theta1_out,theta1_cur)
        theta2_out<-rbind(theta2_out,theta2_cur)
        k_out<-c(k_out,k_cur)
        
    }
    
    theta1_out<-theta1_out[-(1:n_burn),]
    theta2_out<-theta2_out[-(1:n_burn),]
    k_out<-k_out[-(1:n_burn)]
    
    colnames(theta1_out)<-colnames(theta2_out)<-levels(x)
    rownames(theta1_out)<-rownames(theta2_out)<-NULL
    list(theta1_out=theta1_out,theta2_out=theta2_out,k_out=k_out,nx=nx)
}        


rps_ll_vec<-Vectorize(rps_ll,vectorize.args='k')     


k_prior<-function(x,m,v) {
    #neg binom parametrised in terms of mean m and variance v (only called for x between 1 and k_max)
    
    size<-m^2/(v-m)
    dnbinom(x,size=size,mu=m)
}


theta_post<-function(j,rps_gs_out,var_type='win'){
    #takes output of rps_gs and picks up marginal samples 
    #i.e. if j>change point need current theta2, otherwise current theta1
    
    theta<-rps_gs_out$theta1_out
    theta[j>rps_gs_out$k_out]<-rps_gs_out$theta2_out[j>rps_gs_out$k_out]
    theta[,var_type]
}

gs_plot<-function(rps_gs_out,var=c('win','draw','lose')){
    #plots boxplots for marginal posterior of theta_j where y_j ~ MN(theta_j) as function of j
    
    if(length(var)==1){
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type=var)
        boxplot(gs_samples,outline=F,xlab='index',ylab='p')
        title(var)
    }
    else{
        par(mfrow=c(2,2))
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type='win')
        boxplot(gs_samples,outline=F,xlab='index',ylab='p')
        title('win')
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type='draw')
        boxplot(gs_samples,outline=F,xlab='index',ylab='p')
        title('draw')
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type='lose')
        boxplot(gs_samples,outline=F,xlab='index',ylab='p')
        title('lose')
        par(mfrow=c(1,1))
    }
}



rps_1step_pred<-function(rps_gs_out){
    #function to estimate posterior of 1-step predictions
    
    #get samples of theta as appropriate for index nx+1 (i.e. next data point in series)
    win_sample<-theta_post(rps_gs_out$nx+1,rps_gs_out,var_type='win')
    draw_sample<-theta_post(rps_gs_out$nx+1,rps_gs_out,var_type='draw')
    lose_sample<-theta_post(rps_gs_out$nx+1,rps_gs_out,var_type='lose')
    
    #histograms are approx marginal posteriors
    par(mfrow=c(2,2))
    hist(win_sample,prob=T,main='win',xlim=c(0,1),xlab='p')
    hist(draw_sample,prob=T,main='draw',xlim=c(0,1),xlab='p')
    hist(lose_sample,prob=T,main='lose',xlim=c(0,1),xlab='p')
    par(mfrow=c(1,1))
    
    #return estimates of posterior means
    c(win=mean(win_sample),draw=mean(draw_sample),lose=mean(lose_sample))
}

gibbs_plot<-function(rps_gs_out,var=c('A win','Draw','B win')){
    #plots boxplots for marginal posterior of theta_j where y_j ~ MN(theta_j) as function of j
    
    if(length(var)==1){
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type=var)
        gs_samples_df<-do.call(c,gs_samples)
        gs_samples_df<-data.frame(round=rep(1:100,rep(1000,100)),p=gs_samples_df)
        p<-ggplot(data=gs_samples_df,aes(x=round,y=p))+geom_boxplot(aes(group=round),outlier.colour='yellow',fill="red") + 
            scale_y_continuous(limits=c(0,1))+ggtitle(paste0(var," probability posterior distribution")) +theme(plot.title = element_text(hjust = 0.5))
        p
    }
    else{
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type='win')
        gs_samples_df<-do.call(c,gs_samples)
        gs_samples_df<-data.frame(round=rep(1:100,rep(1000,100)),p=gs_samples_df)
        p1<-ggplot(data=gs_samples_df,aes(x=round,y=p))+geom_boxplot(aes(group=round),outlier.colour='yellow',fill="red") + 
            scale_y_continuous(limits=c(0,1))+ggtitle(paste0("A win probability posterior distribution")) +theme(plot.title = element_text(hjust = 0.5))
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type='draw')
        gs_samples_df<-do.call(c,gs_samples)
        gs_samples_df<-data.frame(round=rep(1:100,rep(1000,100)),p=gs_samples_df)
        p2<-ggplot(data=gs_samples_df,aes(x=round,y=p))+geom_boxplot(aes(group=round),outlier.colour='yellow',fill="red") + 
            scale_y_continuous(limits=c(0,1))+ggtitle(paste0("Draw probability posterior distribution")) +theme(plot.title = element_text(hjust = 0.5))
        gs_samples<-lapply(as.list(1:rps_gs_out$nx),theta_post,rps_gs_out=rps_gs_out,var_type='lose')
        gs_samples_df<-do.call(c,gs_samples)
        gs_samples_df<-data.frame(round=rep(1:100,rep(1000,100)),p=gs_samples_df)
        p3<-ggplot(data=gs_samples_df,aes(x=round,y=p))+geom_boxplot(aes(group=round),outlier.colour='yellow',fill="red") + 
            scale_y_continuous(limits=c(0,1))+ggtitle(paste0("B win probability posterior distribution")) +theme(plot.title = element_text(hjust = 0.5))
        grid.arrange(p1,p2,p3,ncol=2)
    }
}


